
/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFAlgnZonesMAZStrokes.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for MAZ grid fitting of uniform-width stroke-based glyphs
 *    Eric Chan and Ron Perry
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The documentation in this file uses the term "real-valued coordinates" to refer
 *    to real numbers such as 0, 1.7, 3/11, and the constant PI. In the floating point
 *    implementation, real-valued coordinates are represented using the ADF_F32
 *    floating point data type. In the fixed point implementation, real-valued
 *    coordinates are represented using the ADF_I1616 fixed point data type.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Algorithm Overview
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The input to the MAZ alignment zone detection and grid fitting algorithm is a
 *    sequence of pen commands representing a uniform-width stroke-based glyph and a
 *    path width (i.e., stroke width). The coordinates of the pen commands and the path
 *    width must be specified in non-negative real-valued image coordinates.
 *
 *    Terminology:
 *
 *      - Given a sequence of pen commands, a lineto command whose endpoints have
 *        identical y coordinates determines a "horizontal segment". Similarly, a
 *        lineto command whose endpoints have identical x coordinates determines a
 *        "vertical segment". The general term "segment" can refer to either a
 *        horizontal segment or a vertical segment. For the sake of brevity, a segment
 *        is often called "horizontal" if it is a horizontal segment. Similarly, a
 *        segment is often called "vertical" if it is a vertical segment.
 *
 *      - Two segments S1 and S2 "overlap" if either (1) S1 and S2 are both horizontal
 *        and there exists a vertical line that intersects both S1 and S2 or (2) S1 and
 *        S2 are both vertical and there exists a horizontal line that intersects both
 *        S1 and S2.
 *
 *      - The "original coordinate" of a horizontal segment H is the original y
 *        coordinate of H in real-valued image coordinates (i.e., the y coordinate
 *        before grid fitting is applied). The "original coordinate" of a vertical
 *        segment V is the original x coordinate of V in real-valued image coordinates
 *        (i.e., the x coordinate before grid fitting is applied).
 *
 *      - The "hinted coordinate" of a horizontal segment H is the y coordinate of H in
 *        real-valued image coordinates after grid fitting has been applied. The
 *        "hinted coordinate" of a vertical segment V is the x coordinate of V in
 *        real-valued image coordinates after grid fitting has been applied.
 *
 *      - Let S1 and S2 be segments with original coordinates c1 and c2, respectively.
 *        S1 and S2 are in "ascending order" if c1 <= c2 and in "descending order" if
 *        c1 >= c2. The terms "ascending order" and "descending order" apply generally
 *        to any number of segments (e.g., 4 segments).
 *
 *      - Let S1, S2, and S3 be segments in ascending order. S2 lies "between" S1 and
 *        S3 if and only if (1) S1 overlaps S2 and (2) S2 overlaps S3.
 *
 *      - A segment C is a "child" or "child segment" of segment S if and only if all
 *        three of the following conditions are satisfied:
 *
 *           a. C's original coordinate <= S's original coordinate,
 *
 *           b. C overlaps S, and
 *
 *           c. no segment lies between C and S.
 *
 *      - Segment S is the "parent" or "parent segment" of segment C if and only if C
 *        is the child of S. If S has multiple child segments, the child segments are
 *        referred to as the "children" of S.
 *
 *      - A segment D is a "descendant" of segment S if and only if (1) D is a child of
 *        S or (2) there exists some ordered sequence of segments S1, S2, ... Sn (where
 *        n >= 1) such that D is a child of S1, S{i} is a child of S{i+1} (where 1 <= i
 *        < n), and Sn is a child of S. If D is a descendant of S, then S is an
 *        "ancestor" of D.
 *
 *      - A "segment tree" is the set of segments defined by a segment S and all
 *        descendants of S. The segment S is the "root segment" of the segment tree.
 *
 *      - A segment S is "rounded up" if S's hinted coordinate h is equal to the
 *        minimum integer greater than S's original coordinate o (i.e., segment S is
 *        rounded up if h == floor(o) + 1). Similarly, S is "rounded down" if S's
 *        hinted coordinate h is equal to the maximum integer not greater than S's
 *        original coordinate o (i.e., segment S is rounded down if h == floor(o)). A
 *        segment tree is rounded up if every segment in the segment tree is rounded
 *        up. Similarly, a segment tree is rounded down if every segment in the segment
 *        tree is rounded down.
 *
 *      - A segment S1 is "aligned" to another segment S2 if S1 has the same hinted
 *        coordinate as S2.
 *
 *      - A segment S is "collapsed" if it has been aligned to at least one of its
 *        child segments. A segment S is "non-collapsed" if it has not been aligned to
 *        any of its child segments.
 *
 *      - Two segments S1 and S2 "collide" if (1) S1 and S2 overlap and (2) their
 *        hinted coordinates are exactly 1 unit apart in real-valued image
 *        coordinates.
 *
 *      - A "near-segment x coordinate" is the x coordinate of a pen command P (i.e.,
 *        P->x, not P->cx) such that (1) the x coordinate coincides with (or lies
 *        within epsilon of) the original coordinate of a vertical segment V and (2)
 *        the y coordinate of P overlaps V in the y direction. A "near-segment y
 *        coordinate" is the y coordinate of a pen command P (i.e., P->y, not P->cy)
 *        such that (1) the y coordinate coincides with (or lies within epsilon of) the
 *        original coordinate of a horizontal segment H and (2) the x coordinate of P
 *        overlaps H in the x direction. More generally, the term "near-segment
 *        coordinate" can refer to either a near-segment x coordinate or a near-segment
 *        y coordinate.
 *
 *      - "Non-segment coordinates" are coordinates of pen commands that are not
 *        "near-segment" coordinates (i.e., the sets of near-segment coordinates and
 *        non-segment coordinates are mutually exclusive and together comprise the
 *        complete set of pen command coordinates).
 *
 *      - A "half-integer" is an element n of the set of real numbers such that n = k +
 *        0.5 for some integer k (e.g., -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, and 3.5 are
 *        all half-integers).
 *
 *      - The "integer pixel grid" is the set of ordered pairs (x,y) of real numbers x
 *        and y such that at least one element of (x,y) is an integer. For example,
 *        (0,0), (1.5,0), and (1.97,3) lie on the integer pixel grid, but (1.7,11.2)
 *        does not. Geometrically, the integer pixel grid is a grid comprising the set
 *        of integer horizontal lines (e.g., y=-2, y=-1, y=0, y=1, and y=2) and the set
 *        of integer vertical lines (e.g., x=-2, x=-1, x=0, x=1, and x=2).
 *
 *      - The "half-integer pixel grid" is the set of ordered pairs (x,y) of real
 *        numbers x and y such that at least one element of (x,y) is a half-integer.
 *        For example, (0,0.5), (1.5,0), and (1.97,2.5) lie on the half-integer pixel
 *        grid, but (1.7,11.2) does not. Geometrically, the half-integer pixel grid is
 *        a grid comprising the set of half-integer horizontal lines (e.g., y=-2.5,
 *        y=-1.5, y=-0.5, y=0.5, y=1.5, and y=2.5) and the set of half-integer vertical
 *        lines (e.g., x=-2.5, x=-1.5, x=-0.5, x=0.5, x=1.5, and x=2.5).
 *
 *      - The "minimum anchor segment of an x coordinate" is the non-collapsed vertical
 *        segment with the maximum original coordinate that is less than or equal to x
 *        and the "maximum anchor segment of an x coordinate" is the non-collapsed
 *        vertical segment with the minimum original coordinate that is greater than or
 *        equal to x. Similarly, the "minimum anchor segment of a y coordinate" is the
 *        non-collapsed horizontal segment with the maximum original coordinate that is
 *        less than or equal to y and the "maximum anchor segment of a y coordinate" is
 *        the non-collapsed horizontal segment with the minimum original coordinate
 *        that is greater than or equal to y. During grid fitting, non-segment
 *        coordinates are linearly interpolated between their corresponding anchor
 *        segments; non-segment coordinates that are missing at least one anchor
 *        segment are treated as a special case by using the minimum or maximum edge of
 *        the glyph's bounding box as a "virtual" anchor segment (where the "minimum"
 *        and "maximum" edges correspond to the left and right edges of the bounding
 *        box for non-segment x coordinates, respectively, and where the "minimum" and
 *        "maximum" edges correspond to the bottom and top edges of the bounding box
 *        for non-segment y coordinates, respectively).
 *
 *    The following general notes apply to the algorithm described below:
 *
 *      - The algorithm is invoked dynamically during rendering. Therefore, the
 *        algorithm is designed to achieve a favorable balance between high quality,
 *        small memory consumption, and computational efficiency.
 *
 *      - The algorithm can be invoked on any uniform-width stroke-based glyph, but it
 *        is designed and optimized for CJK glyphs.
 *
 *      - Uniform-width stroke-based glyphs are largely comprised of horizontal and
 *        vertical strokes. Therefore, the algorithm detects horizontal and vertical
 *        features of the glyph and aligns them to the sampling grid. The algorithm
 *        does not make any effort to align other features (e.g., curves or diagonal
 *        lines) to the sampling grid. Coordinates that are not aligned to the sampling
 *        grid are instead hinted using interpolation (see below for details). This
 *        overall strategy balances the competing goals of high quality and high
 *        runtime performance.
 *
 *      - The algorithm detects and resolves "collisions" between strokes, thereby
 *        preventing important features from becoming visually indistinguishable (see
 *        below for details). Collision resolution plays a significant role in the
 *        quality of the results. The algorithm uses a greedy method to resolve
 *        collisions. This method is computationally efficient but may not always find
 *        the optimal grid fitting configuration. In practice, however, this method
 *        produces high-quality results across a wide range of glyphs.
 *
 *      - Collision resolution is designed and optimized for small to moderate ppems
 *        (30 ppems and below). For most uniform-width stroke-based glyphs, the path
 *        width is expressed as a small fraction of the em box (e.g., 5%). When
 *        rendering these glyphs at small to moderate ppems, the hinted path width will
 *        be 1 pixel (e.g., 24 ppem * 0.05 = 1.2 pixels, which rounds down to 1 pixel).
 *        Consequently, collisions are defined above assuming a hinted path width of 1
 *        pixel (i.e., two overlapping segments "collide" if their hinted coordinates
 *        are exactly 1 pixel apart). Collision resolution is rarely needed at larger
 *        ppems (above 30 ppems) because more pixels are available to represent each
 *        feature of the glyph clearly.
 *
 *      - During collision resolution, the algorithm does not attempt to preserve the
 *        original spacing or the original spacing proportions between segments.
 *
 *      - Ultimately, grid fitting requires modifying the original coordinates of the
 *        pen commands. To minimize the distortion of the glyph's shape, the algorithm
 *        never moves a coordinate more than 1.5 pixels away from its original value.
 *
 *      - Steps 1 through 5 of the algorithm (see below) perform grid fitting using the
 *        internal MAZStrokeSegment data structure. No changes are made to the original
 *        pen commands until Step 6.
 *
 *      - The algorithm hints the path width by rounding the original path width to the
 *        nearest integer and clamping it to at least 1 pixel. If the hinted path width
 *        is an even integer, stroke skeletons are aligned to the half-integer pixel
 *        grid. If the hinted path width is an odd integer, stroke skeletons are
 *        aligned to the integer pixel grid. This convention ensures that the edges of
 *        the rendered glyph are positioned on the sampling grid, thereby maximizing
 *        edge contrast.
 *
 *      - To optimize performance and to maximize code re-use between floating point
 *        and fixed point implementations, most of the steps of the algorithm perform
 *        grid fitting using the integer pixel grid (regardless of whether the hinted
 *        path width is even or odd). Adjustments to the hinted coordinates to
 *        accommodate the hinted path width are delayed until the final step of the
 *        algorithm (see Step 8 below).
 *
 *      - Steps 1 through 7 of the algorithm described below are effectively applied
 *        twice in this implementation -- once for horizontal segments and once for
 *        vertical segments. Note that this algorithm contains no dependencies between
 *        horizontal segments and vertical segments; consequently, hardware and
 *        multi-core implementations can process horizontal segments and vertical
 *        segments completely in parallel.
 *
 *    Algorithm description:
 *
 *      1. Traverse the input pen commands to identify segments, which are represented
 *         by the MAZStrokeSegment data structure, and store the segments in an
 *         unordered segment array.
 *
 *      2. Compute the hinted coordinate of each segment in the segment array by
 *         rounding the original coordinate to the nearest integer. This hinted
 *         coordinate may be modified below to resolve collisions (see Step 5) and to
 *         accommodate the hinted path width (see Step 8).
 *
 *      3. Sort the segments in the segment array into ascending order (i.e., sort by
 *         non-decreasing original coordinates).
 *
 *      4. Identify the child segments of each segment in the segment array and build a
 *         directed acyclic graph (DAG). The DAG is represented implicitly by the
 *         MAZStrokeSegment data structure by linking each segment to its child
 *         segments. The purpose of building this DAG is to detect and resolve
 *         collisions between segments in the next step. A collision between a segment
 *         S and a child segment C cannot occur during collision resolution if the
 *         current hinted coordinates of S and C differ by more than 3 pixels.
 *         Therefore, to improve runtime efficiency, a link in the DAG is added between
 *         S and C if and only if their hinted coordinates differ by 3 pixels or less.
 *
 *      5. Detect colliding segments and resolve collisions that occur in Step 2 using
 *         a greedy bottom-up algorithm. Two overlapping segments collide if their
 *         hinted coordinates lie exactly 1 pixel apart. Visually, colliding segments
 *         appear to be a 2-pixel wide stroke (instead of two separate 1-pixel wide
 *         strokes). Collisions can be resolved in some cases by adding +1 or -1 to the
 *         hinted coordinate of one of the segments so that the two segments become at
 *         least 2 pixels apart (see details below).
 *
 *         Collisions are resolved using a greedy algorithm in which the DAG of
 *         segments determined in the previous step is traversed in a bottom-up order
 *         (i.e., from the bottommost descendants to the topmost ancestors). Since the
 *         segments are sorted in ascending order (see Step 3 above), this bottom-up
 *         traversal is accomplished by visiting each segment in the segment array,
 *         beginning with the first segment in the segment array and ending with the
 *         last segment in the segment array. For each segment S visited during this
 *         bottom-up traversal, apply the following steps for each child C of S that
 *         collides with S:
 *
 *           a. If C is rounded up and can be rounded down without colliding with any
 *              of its children, then round down C and jump to Step (e). (Example: if
 *              C's original coordinate is 3.8, then C's hinted coordinate is normally
 *              rounded up to the nearest integer 4.0. Rounding down instead produces
 *              the integer 3.0.) This step attempts to eliminate a collision and
 *              always avoids introducing any new collisions.
 *
 *           b. If C is rounded up and the segment tree with root C can be rounded down
 *              without creating any collisions whatsoever in the segment tree, then
 *              round down the segment tree and jump to Step (e). This step attempts to
 *              eliminate a collision and always avoids introducing any new
 *              collisions.
 *
 *           c. If S is rounded down, then round up S and jump to Step (e). (Example:
 *              if S's original coordinate is 5.4, then S's hinted coordinate is
 *              normally rounded down to the nearest integer 5.0. Rounding up instead
 *              produces the integer 6.0.) This step attempts to eliminate a collision
 *              but may introduce a collision between S and a parent P of S. This
 *              potential collision will be treated when processing segment P in a
 *              subsequent iteration of the bottom-up traversal of the DAG.
 *
 *           d. Align S to C. (Example: if C's hinted coordinate is 4.0 and S's hinted
 *              coordinate is 5.0, then align S to 4.0.)
 *
 *           e. Done.
 *
 *         Steps (a) through (c) attempt to round segments C and S so that they are at
 *         least 2 pixels apart. However, there are some cases in which collisions
 *         cannot be resolved by this method. Since C and S cannot be visually
 *         distinguished in these cases, Step (d) aligns S to C, thereby simplifying
 *         the appearance of the glyph by aligning two segments to the same coordinate.
 *
 *      6. Perform grid fitting on near-segment coordinates and store the results in
 *         the original pen commands. x coordinates of pen commands that lie close to a
 *         vertical segment and overlap the vertical segment in the y direction are
 *         aligned to the vertical segment. Similarly, y coordinates of pen commands
 *         that lie close to a horizontal segment and overlap the horizontal segment in
 *         the x direction are aligned to the horizontal segment.
 *
 *      7. Perform grid fitting on non-segment coordinates and store the results in the
 *         original pen commands. Non-segment x coordinates of pen commands are
 *         linearly interpolated in the x direction between the minimum and maximum
 *         vertical anchor segments around x. Similarly, non-segment y coordinates of
 *         pen commands are linearly interpolated in the y direction between the
 *         minimum and maximum horizontal anchor segments around y.
 *
 *      8. Compute the hinted path width by rounding the original path width to the
 *         nearest integer. Ensure that the hinted path width is at least 1 pixel. If
 *         the hinted path width is an even integer, add 0.5 to each coordinate of each
 *         pen command. This step effectively aligns stroke skeletons to the
 *         half-integer pixel grid.
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
#include <stdlib.h>

#include "fs_object.h"
#include "fs_function.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: iType ADF Rendering
 *-----------------------------------------------------------------------------------
 */
#ifdef FS_EDGE_HINTS

/*#include "adfinittermsystem.h"*/  /* not needed */
#include "adfalgnzonesmaz.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: IMPLICIT ADFS ONLY
 *-----------------------------------------------------------------------------------
 */
#if (ADF_USE_IMPLICIT_ADFS)


/**
 *-----------------------------------------------------------------------------------
 *    START: MAZ DETECTION AND GRID FITTING FOR UNIFORM-WIDTH STROKE-BASED GLYPHS ONLY
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    START: FIXED POINT MATH ONLY
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Constants for the ADF_I1616 fixed point values of 1, 0.5, and 0.2
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_ONE                 (I1616_CONST_1)
#define MAZ_STROKE_ONE_HALF             (I1616_CONST_ONE_HALF)
#define MAZ_STROKE_ONE_FIFTH         (I1616_CONST_ONE_FIFTH)


/**
 *-----------------------------------------------------------------------------------
 *    During the hinting of the path width in MAZStrokeAutohintPathWidth(), an original
 *    path width whose fractional portion is less than MAZ_MAX_SAFETY_THRESHOLD will be
 *    rounded down. MAZ_MAX_SAFETY_THRESHOLD is mathematically equivalent to 33280.0 /
 *    65536.0. This specific value was chosen (empirically) because it allows
 *    sufficient margin for numerical round-off error and can be represented exactly as
 *    a floating point value and as an ADF_I1616 fixed point value. See the
 *    implementation of MAZStrokeAutohintPathWidth() below for a detailed explanation of
 *    why it is insufficient to hint the path width by simply rounding the original
 *    path width to the nearest integer.
 *-----------------------------------------------------------------------------------
 */
#define MAZ_MAX_SAFETY_THRESHOLD     (0x00008200)


/**
 *-----------------------------------------------------------------------------------
 *    Round the non-negative ADF_I1616 fixed point value v to the nearest integer
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_ROUND_INT(v)         (((v) + I1616_CONST_ONE_HALF) >> 16)


/**
 *-----------------------------------------------------------------------------------
 *    Round the non-negative ADF_I1616 fixed point value v to the maximum integral
 *    ADF_I1616 fixed point value not greater than v
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_FLOOR(v)             ((v) & 0xffff0000)


/**
 *-----------------------------------------------------------------------------------
 *    Return true (i.e., a non-zero value) if the ADF_I1616 fixed point value v (when
 *    truncated) is even
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_IS_EVEN(v)         (!((v) & 0x10000))


/**
 *-----------------------------------------------------------------------------------
 *    Convert v from an integer to an ADF_I1616 fixed point value
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_INT_TO_R32(v)     ((v) << 16)


/**
 *-----------------------------------------------------------------------------------
 *    Perform linear interpolation in two steps and return the interpolated result.
 *    First, a ratio t is determined by (vOld - aOld) / (bOld - aOld). Next, the ratio
 *    t is used to perform linear interpolation between aNew and bNew.
 *-----------------------------------------------------------------------------------
 */
ADF_INLINE static ADF_R32 MAZStrokeLerp (ADF_R32 vOld, ADF_R32 aOld, ADF_R32 bOld,
ADF_R32 aNew, ADF_R32 bNew)
{
    ADF_I32 status;
    ADF_R32 t = I1616_DIV(vOld - aOld, bOld - aOld, &status);
    return(aNew + I1616_MUL(t, bNew - aNew));
}


/**
 *-----------------------------------------------------------------------------------
 *    END: FIXED POINT MATH ONLY
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Stroke Alignment Status Bits
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    To save memory, this implementation temporarily uses the high-order bits of the
 *    MAZPenCmd's opCode element to record status information. These high-order bits
 *    are normally clear (i.e., set to zero) and are not used.
 *
 *    The opCode element is defined in the MAZPenCmd data structure as an unsigned
 *    32-bit integer (see the ADFPenCmd data structure in ADFTypeSystem.h and the
 *    ADFPenCmdFx data structure in ADFAlgnZonesMAZ.h for details). Changes to the
 *    opCode definition (e.g., changing the data type from an unsigned 32-bit integer
 *    to an unsigned 16-bit integer) must be reflected appropriately in the macro
 *    definitions below.
 *
 *    The MAZ_STROKE_X_ALIGNED bit is set if a pen command's x coordinate is aligned to
 *    the sampling grid. Similarly, the MAZ_STROKE_Y_ALIGNED bit is set if a pen
 *    command's y coordinate is aligned to the sampling grid. MAZ_STROKE_SNAP_MASK is
 *    used to clear these two bits.
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_X_ALIGNED (0x80000000)
#define MAZ_STROKE_Y_ALIGNED (0x40000000)


/**
 *-----------------------------------------------------------------------------------
 *    Identify horizontal and vertical segments. Upon entry, penCmds is a contiguous
 *    array of numPenCmds pen commands whose coordinates are specified in non-negative
 *    real-valued image coordinates. hSegs and vSegs are uninitialized contiguous
 *    arrays of MAZStrokeSegments, where each array contains at least numPenCmds
 *    elements. Upon exit, hSegs contains a valid array of horizontal segments, vSegs
 *    contains a valid array of vertical segments, outNumHSegments contains the number
 *    of horizontal segments in the hSegs array, and outNumVSegments contains the
 *    number of vertical segments in the vSegs array.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeFindSegmentsBottomUp (fnt_LocalGraphicStateType *gs,
                                               MAZStrokeSegment *hSegs,
                                               MAZStrokeSegment *vSegs,
                                               ADF_I32 *outNumHSegments,
                                               ADF_I32 *outNumVSegments)
{
    /*----Set a pointer to the first (uninitialized) horizontal segment. Set a
     *----pointer to the first (uninitialized) vertical segment.*/
    MAZStrokeSegment *hSeg = hSegs;
    MAZStrokeSegment *vSeg = vSegs;


    /*----Set a pointer to the first pen command and a pointer to the previous
     *----(initially invalid) pen command */


    ADF_I32 i;
    ADF_I32 j;

    ADF_R32 xj,yj;

    /*----Process each pen command*/

    for (i = 0; i < gs->CE0->nc; ++i) {


      for (j=gs->CE0->sp[i]+1; j<=gs->CE0->ep[i]; j++) {



        ADF_I32 nextj = j+1;
        if (nextj > gs->CE0->ep[i])
            nextj = gs->CE0->sp[i]+1;


        /*----p is a lineto command. Check for a horizontal segment. */
        if (( gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] &&
            gs->CE0->y[j] == gs->CE0->y[j-1] ) ||

            /* top of curve tangent */
            (  gs->CE0->y[j] == gs->CE0->y[j-1]
            && gs->CE0->y[j] == gs->CE0->y[nextj] && nextj != j-1 &&
            gs->CE0->onCurve[j] && !gs->CE0->onCurve[j-1] && !gs->CE0->onCurve[nextj]) ||

            /* quad which degenerated into a line */
            (  gs->CE0->y[j] == gs->CE0->y[j-1]
            && gs->CE0->y[j] == gs->CE0->y[nextj] && nextj > j &&
            !gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] && gs->CE0->onCurve[nextj])
            ) {

            hSeg->hintFlags = 0;


            /*----A horizontal segment has been found. Set its original y coordinate.*/
            hSeg->cOrig = gs->CE0->y[j];


            /*----Initialize isCollapsed to false and the number of children to zero*/
            hSeg->isCollapsed = 0;
            hSeg->numChildren = 0;
            hSeg->numParents = 0;


            /*----Set cHint to the integer nearest to cOrig*/
            hSeg->cHint = MAZ_STROKE_ROUND_INT(hSeg->cOrig);


            /**
             *----Set cHintLow to the maximum integer not greater than cOrig (i.e.,
             *----set cHintLow to the "floor" of cOrig)
             */
            hSeg->cHintLow = (MAZ_STROKE_INT_TO_R32(hSeg->cHint) <= hSeg->cOrig) ?
            (hSeg->cHint) : (hSeg->cHint - 1);



            /*----Determine the segment's minimum and maximum x values */
            xj = gs->CE0->x[j];
            if (!gs->CE0->onCurve[j-1] || !gs->CE0->onCurve[j])
                xj = gs->CE0->x[nextj];
            if (xj < gs->CE0->x[j-1]) {
                hSeg->cMin = xj; /*gs->CE0->x[j];*/
                   hSeg->cMax = gs->CE0->x[j-1];
            } else {
                hSeg->cMin = gs->CE0->x[j-1];
                hSeg->cMax = xj; /*gs->CE0->x[j];*/
            }

            /**
            *----Set collapsePriority
            */
            if (gs->CE0->f[j] & 1)
                hSeg->hintFlags |= HF_HREDUCE;


            /*----Advance the horizontal segment pointer to the next (uninitialized)
             *----horizontal segment*/
            ++hSeg;
        }


        /*----Check for a vertical segment */
        if ( (gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] &&
            gs->CE0->x[j] == gs->CE0->x[j-1]) ||

            /* top of curve tangent */

            ( gs->CE0->x[j] == gs->CE0->x[j-1]
            && gs->CE0->x[j] == gs->CE0->x[nextj] && nextj != j-1 &&
            gs->CE0->onCurve[j] && !gs->CE0->onCurve[j-1] && !gs->CE0->onCurve[nextj]) ||

            /* quad which degenerated into a line */
            ( gs->CE0->x[j] == gs->CE0->x[j-1]
            && gs->CE0->x[j] == gs->CE0->x[nextj] && nextj > j &&
            !gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] && gs->CE0->onCurve[nextj])
            ) {

            vSeg->hintFlags = 0;


            /*----A vertical segment has been found. Set its original x coordinate.*/
            vSeg->cOrig = gs->CE0->x[j];


            /*----Initialize isCollapsed to false and the number of children to zero*/
            vSeg->isCollapsed = 0;
            vSeg->numChildren = 0;
            vSeg->numParents = 0;


            /*----Set cHint to the integer nearest to cOrig*/
            vSeg->cHint = MAZ_STROKE_ROUND_INT(vSeg->cOrig);


            /**
             *----Set cHintLow to the maximum integer not greater than cOrig (i.e.,
             *----set cHintLow to the "floor" of cOrig)
             */
            vSeg->cHintLow = (MAZ_STROKE_INT_TO_R32(hSeg->cHint) <= vSeg->cOrig) ?
            (vSeg->cHint) : (vSeg->cHint - 1);


            /*----Determine the segment's minimum and maximum y values */
            yj = gs->CE0->y[j];
            if (!gs->CE0->onCurve[j-1] || !gs->CE0->onCurve[j])
                yj = gs->CE0->y[nextj];
            if (yj < gs->CE0->y[j-1]) {
                vSeg->cMin = yj; /*gs->CE0->y[j];*/
                vSeg->cMax = gs->CE0->y[j-1];
            } else {
                vSeg->cMin = gs->CE0->y[j-1];
                vSeg->cMax = yj; /*gs->CE0->y[j];*/
            }

            /**
            *----Set collapsePriority
            */
            if (gs->CE0->f[j] & 2)
                vSeg->hintFlags |= HF_VREDUCE;


            /*----Advance the vertical segment pointer to the next (uninitialized)
             *----vertical segment*/
            ++vSeg;
        }
      }
    }


    /*----Store the number of identified horizontal segments and the number of
     *----identified vertical segments*/
    *outNumHSegments = hSeg - hSegs;
    *outNumVSegments = vSeg - vSegs;
}

/**
 *-----------------------------------------------------------------------------------
 *    Identify horizontal and vertical segments. Upon entry, penCmds is a contiguous
 *    array of numPenCmds pen commands whose coordinates are specified in non-negative
 *    real-valued image coordinates. hSegs and vSegs are uninitialized contiguous
 *    arrays of MAZStrokeSegments, where each array contains at least numPenCmds
 *    elements. Upon exit, hSegs contains a valid array of horizontal segments, vSegs
 *    contains a valid array of vertical segments, outNumHSegments contains the number
 *    of horizontal segments in the hSegs array, and outNumVSegments contains the
 *    number of vertical segments in the vSegs array.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeFindSegmentsTopDown (fnt_LocalGraphicStateType *gs,
                                              MAZStrokeSegment *hSegs,
                                              MAZStrokeSegment *vSegs,
                                              ADF_I32 *outNumHSegments,
                                              ADF_I32 *outNumVSegments)
{
    /*----Set a pointer to the first (uninitialized) horizontal segment. Set a
     *----pointer to the first (uninitialized) vertical segment.*/
    MAZStrokeSegment *hSeg = hSegs;
    MAZStrokeSegment *vSeg = vSegs;


    /*----Set a pointer to the first pen command and a pointer to the previous
     *----(initially invalid) pen command*/


    ADF_I32 i;
    ADF_I32 j;

    ADF_R32 xj,yj;

    /*----Process each pen command*/

    for (i = 0; i < gs->CE0->nc; ++i) {


        for (j=gs->CE0->sp[i]+1; j<=gs->CE0->ep[i]; j++) {


            ADF_I32 nextj = j+1;
            if (nextj > gs->CE0->ep[i])
                nextj = gs->CE0->sp[i]+1;


            /*----p is a lineto command. Check for a horizontal segment. */
            if (( gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] &&
                 gs->CE0->y[j] == gs->CE0->y[j-1] ) ||

                /* top of curve tangent */
                (  gs->CE0->y[j] == gs->CE0->y[j-1]
                && gs->CE0->y[j] == gs->CE0->y[nextj] && nextj != j-1 &&
                gs->CE0->onCurve[j] && !gs->CE0->onCurve[j-1] && !gs->CE0->onCurve[nextj]) ||

                /* quad which degenerated into a line */
                (  gs->CE0->y[j] == gs->CE0->y[j-1]
                && gs->CE0->y[j] == gs->CE0->y[nextj] && nextj > j &&
                !gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] && gs->CE0->onCurve[nextj])
                )  {

                hSeg->hintFlags = 0;

                /*----A horizontal segment has been found. Set its original y coordinate.*/
                hSeg->cOrig = gs->CE0->y[j];


                /*----Initialize isCollapsed to false and the number of children to zero*/
                hSeg->isCollapsed = 0;
                hSeg->numChildren = 0;
                hSeg->numParents = 0;


                /*----Set cHint to the integer nearest to cOrig*/
                hSeg->cHint = MAZ_STROKE_ROUND_INT(hSeg->cOrig);


                /*----Set cHintLow to the maximum integer not greater than cOrig (i.e.,
                *----set cHintLow to the "floor" of cOrig)*/
                hSeg->cHintLow = (MAZ_STROKE_INT_TO_R32(hSeg->cHint) >= hSeg->cOrig) ?
                (hSeg->cHint) : (hSeg->cHint + 1);


                /*----Determine the segment's minimum and maximum x values */
                xj = gs->CE0->x[j];
                if (!gs->CE0->onCurve[j-1] || !gs->CE0->onCurve[j])
                    xj = gs->CE0->x[nextj];
                if (xj < gs->CE0->x[j-1]) {
                    hSeg->cMin = xj; /*gs->CE0->x[j];*/
                       hSeg->cMax = gs->CE0->x[j-1];
                } else {
                    hSeg->cMin = gs->CE0->x[j-1];
                    hSeg->cMax = xj; /*gs->CE0->x[j];*/
                }

                /**
                *----Set collapsePriority
                */
                if ((gs->CE0->f[j] & 1) || (gs->CE0->f[j-1] & 1))
                    hSeg->hintFlags |= HF_HREDUCE;


                /*----Advance the horizontal segment pointer to the next (uninitialized)
                *----horizontal segment*/
                ++hSeg;
            }


            /*----Check for a vertical segment */
            if ( (gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] &&
                gs->CE0->x[j] == gs->CE0->x[j-1]) ||

                /* top of curve tangent */

                ( gs->CE0->x[j] == gs->CE0->x[j-1]
                && gs->CE0->x[j] == gs->CE0->x[nextj] && nextj != j-1 &&
                gs->CE0->onCurve[j] && !gs->CE0->onCurve[j-1] && !gs->CE0->onCurve[nextj]) ||

                /* quad which degenerated into a line */
                ( gs->CE0->x[j] == gs->CE0->x[j-1] 
                && gs->CE0->x[j] == gs->CE0->x[nextj] && nextj > j &&
                !gs->CE0->onCurve[j] && gs->CE0->onCurve[j-1] && gs->CE0->onCurve[nextj])
                ) {

                vSeg->hintFlags = 0;


                /*----A vertical segment has been found. Set its original x coordinate.*/
                vSeg->cOrig = gs->CE0->x[j];


                /*----Initialize isCollapsed to false and the number of children to zero*/
                vSeg->isCollapsed = 0;
                vSeg->numChildren = 0;
                vSeg->numParents = 0;


                /*----Set cHint to the integer nearest to cOrig*/
                vSeg->cHint = MAZ_STROKE_ROUND_INT(vSeg->cOrig);


                /*----Set cHintLow to the maximum integer not greater than cOrig (i.e.,
                *----set cHintLow to the "floor" of cOrig)*/
                vSeg->cHintLow = (MAZ_STROKE_INT_TO_R32(vSeg->cHint) >= vSeg->cOrig) ?
                (vSeg->cHint) : (vSeg->cHint + 1);


                /*----Determine the segment's minimum and maximum y values */
                yj = gs->CE0->y[j];
                if (!gs->CE0->onCurve[j-1] || !gs->CE0->onCurve[j])
                    yj = gs->CE0->y[nextj];
                if (yj < gs->CE0->y[j-1]) {
                    vSeg->cMin = yj; /*gs->CE0->y[j];*/
                    vSeg->cMax = gs->CE0->y[j-1];
                } else {
                    vSeg->cMin = gs->CE0->y[j-1];
                    vSeg->cMax = yj; /*gs->CE0->y[j];*/
                }

                /**
                *----Set collapsePriority
                */
                if ( (gs->CE0->f[j] & 2) || (gs->CE0->f[j-1] & 2))
                    vSeg->hintFlags |= HF_VREDUCE;


                /*----Advance the vertical segment pointer to the next (uninitialized)
                *----vertical segment*/
                ++vSeg;
            }
        }
    }


    /*----Store the number of identified horizontal segments and the number of
     *----identified vertical segments*/
    *outNumHSegments = hSeg - hSegs;
    *outNumVSegments = vSeg - vSegs;
}
/**
 *-----------------------------------------------------------------------------------
 *    Sort the specified segment array segs into ascending order (i.e., sort by
 *    non-decreasing original coordinates) using the selection sort algorithm. Upon
 *    entry, segs is a contiguous array of numSegs segments. The segs array is sorted
 *    in-place.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeSortSegmentsBottomUp (MAZStrokeSegment *segs, ADF_I32 numSegs)
{
    /**
     *----Set a pointer to the first segment
     */
    MAZStrokeSegment *s = segs;


    /**
     *----Process each segment except for the last segment
     */
    ADF_I32 i;
    for (i = 0; i < numSegs - 1; ++i, ++s) {


        /**
         *----Initialize the minimum original coordinate
         */
        ADF_R32 cMin = s->cOrig;
        ADF_I32 j;


        /**
         *----Initialize a pointer to the segment with the minimum original
         *----coordinate
         */
        MAZStrokeSegment *minSeg = s;


        /**
         *----Set a pointer to the segment after s
         */
        MAZStrokeSegment *sn = s + 1;


        /**
         *----Process each segment after s
         */
        for (j = i + 1; j < numSegs; ++j, ++sn) {


            /**
             *----If segment sn has an original coordinate that is less than the
             *----current minimum original coordinate (i.e., cMin), update cMin and
             *----minSeg
             */
            if (sn->cOrig < cMin) {
                cMin = sn->cOrig;
                minSeg = sn;
            }
        }


        /**
         *----If the segment with the minimum original coordinate value is different
         *----from s, swap the values in s and minSeg
         */
        if (minSeg != s) {
            MAZStrokeSegment tmp = *minSeg;
            *minSeg = *s;
            *s = tmp;
        }
    }
}

/*-----------------------------------------------------------------------------------
 *  Sort the specified segment array segs into ascending order (i.e., sort by
 *  non-decreasing original coordinates) using the selection sort algorithm. Upon
 *  entry, segs is a contiguous array of numSegs segments. The segs array is sorted
 *  in-place.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeSortSegmentsTopDown (MAZStrokeSegment *segs, ADF_I32 numSegs)
{
    /*----Set a pointer to the first segment*/
    MAZStrokeSegment *s = segs;


    /*----Process each segment except for the last segment*/
    ADF_I32 i;
    for (i = 0; i < numSegs - 1; ++i, ++s) {


        /*----Initialize the minimum original coordinate*/
        ADF_R32 cMin = s->cOrig;
        ADF_I32 j;


        /*----Initialize a pointer to the segment with the minimum original
         *----coordinate*/
        MAZStrokeSegment *minSeg = s;


        /*----Set a pointer to the segment after s*/
        MAZStrokeSegment *sn = s + 1;


        /*----Process each segment after s*/
        for (j = i + 1; j < numSegs; ++j, ++sn) {


            /*----If segment sn has an original coordinate that is less than the
             *----current minimum original coordinate (i.e., cMin), update cMin and
             *----minSeg
            */
            if (sn->cOrig > cMin) {
                cMin = sn->cOrig;
                minSeg = sn;
            }
        }


        /*----If the segment with the minimum original coordinate value is different
         *----from s, swap the values in s and minSeg
        */
        if (minSeg != s) {
            MAZStrokeSegment tmp = *minSeg;
            *minSeg = *s;
            *s = tmp;
        }
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    Return true (i.e., a non-zero value) if segments s1 and s2 overlap by at least
 *    MAZ_STROKE_ONE_FIFTH units in real-valued image coordinates. Return false (i.e.,
 *    zero) otherwise.
 *-----------------------------------------------------------------------------------
 */
ADF_INLINE static ADF_I32 MAZStrokeSegsOverlap (MAZStrokeSegment *s1,
MAZStrokeSegment *s2)
{
    return(!((s1->cMax - MAZ_STROKE_ONE_FIFTH < s2->cMin) ||
    (s2->cMax - MAZ_STROKE_ONE_FIFTH < s1->cMin)));
}


/**
 *-----------------------------------------------------------------------------------
 *    Build a directed acyclic graph (DAG) of segments from the specified segment array
 *    segs. The DAG is represented implicitly in segs by linking each segment to its
 *    child segments. Upon entry, segs is a contiguous array of numSegs segments in
 *    ascending order.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeBuildDagBottomUp (MAZStrokeSegment *segs, ADF_I32 numSegs)
{
    /**
     *----Set a pointer to the second segment (skip the first segment because the
     *----first segment has the minimum original coordinate and therefore cannot have
     *----any children)
     */
    MAZStrokeSegment *s = segs + 1;


    /**
     *----Process each segment in ascending order
     */
    ADF_I32 i;
    for (i = 1; i < numSegs; ++i, ++s) {


        /**
         *----Set a pointer to the previous segment
         */
        MAZStrokeSegment *sp = s - 1;


        /**
         *----Process the previous segments in descending order
         */
        ADF_I32 j;
        for (j = i - 1; j >= 0; --j, --sp) {


            /**
             *----Initialize the addChild Boolean to true (i.e., initially assume
             *----that segment sp will be added as a child to segment s)
             */
            ADF_I32 addChild = 1;
            ADF_I32 k;


            /**
             *----If the difference between s's current hinted coordinate and sp's
             *----current hinted coordinate is greater than 3 pixels, then s and sp
             *----cannot collide. In this case, exit the inner for-loop to prevent sp
             *----from being added as a child of s, thereby improving overall runtime
             *----efficiency.
             */
            if (s->cHint - sp->cHint > 3) break;


            /**
             *----If segments s and sp do not overlap, proceed to the next segment
             */
            if (!MAZStrokeSegsOverlap(s, sp)) continue;


            /**
             *----Segments s and sp overlap. Determine if sp overlaps any of the
             *----children of s. If an overlap occurs, set the addChild Boolean to
             *----false and exit the loop.
             */
            for (k = 0; k < s->numChildren; ++k) {
                if (MAZStrokeSegsOverlap(sp, s->children[k])) {
                    addChild = 0;
                    break;
                }
            }


            /**
             *----If the addChild Boolean is true (i.e., sp does not overlap any of
             *----the children of s), add sp as a child of s
             */
            if (addChild) {


                /**
                 *----Add sp as a child of s and update the number of children of s
                 */
                s->children[s->numChildren++] = sp;
                sp->parents[sp->numParents++] = s;


                /**
                 *----If s has reached the maximum number of children, proceed to the
                 *----next segment (i.e., do not add any more children to s)
                 */
                if (s->numChildren >= MAZ_STROKE_MAX_CHILDREN) break;
            }
        }
    }
}

static ADF_Void MAZStrokeBuildDagTopDown (MAZStrokeSegment *segs, ADF_I32 numSegs)
{
    /*----Set a pointer to the second segment (skip the first segment because the
     *----first segment has the minimum original coordinate and therefore cannot have
     *----any children)
    */
    MAZStrokeSegment *s = segs + 1;


    /*----Process each segment in ascending order*/
    ADF_I32 i;
    for (i = 1; i < numSegs; ++i, ++s) {


        /*----Set a pointer to the previous segment*/
        MAZStrokeSegment *sp = s - 1;


        /*----Process the previous segments in descending order*/
        ADF_I32 j;
        for (j = i - 1; j >= 0; --j, --sp) {


            /*----Initialize the addChild Boolean to true (i.e., initially assume
             *----that segment sp will be added as a child to segment s)
            */
            ADF_I32 addChild = 1;
            ADF_I32 k;


            /*----If the difference between s's current hinted coordinate and sp's
             *----current hinted coordinate is greater than 3 pixels, then s and sp
             *----cannot collide. In this case, exit the inner for-loop to prevent sp
             *----from being added as a child of s, thereby improving overall runtime
             *----efficiency.
             */
            if (sp->cHint - s->cHint > 3) break;


            /*----If segments s and sp do not overlap, proceed to the next segment*/
            if (!MAZStrokeSegsOverlap(s, sp)) continue;


            /*----Segments s and sp overlap. Determine if sp overlaps any of the
             *----children of s. If an overlap occurs, set the addChild Boolean to
             *----false and exit the loop.
            */
            for (k = 0; k < s->numChildren; ++k) {
                if (MAZStrokeSegsOverlap(sp, s->children[k])) {
                    addChild = 0;
                    break;
                }
            }


            /*----If the addChild Boolean is true (i.e., sp does not overlap any of
             *----the children of s), add sp as a child of s
             */
            if (addChild) {


                /*----Add sp as a child of s and update the number of children of s*/
                s->children[s->numChildren++] = sp;
                sp->parents[sp->numParents++] = s;


                /*----If s has reached the maximum number of children, proceed to the
                 *----next segment (i.e., do not add any more children to s)
                 */
                if (s->numChildren >= MAZ_STROKE_MAX_CHILDREN) break;
            }
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    Round down the segment tree whose root is s
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeRoundDownTree (MAZStrokeSegment *s)
{
    /**
     *----Initialize last to s. last represents a pointer to the last element in the
     *----linked list L (see below).
     */
    MAZStrokeSegment *last = s;


    /**
     *----Initialize the link pointer of s to NULL. link is used below to implement a
     *----linked list of segments (denoted L below) for performing a breadth-first
     *----traversal of the segment tree whose root is s.
     */
    s->link = 0;


    /**
     *----Process the segment tree using a breadth-first traversal of L
     */
    for (; s; s = s->link) {


        /**
         *----If s is the root of a segment tree that cannot be rounded down (i.e.,
         *----the segment tree has already been rounded down and therefore cannot be
         *----rounded down again), then proceed to the next segment
         */
        ADF_I32 i;
        if (!(s->canRoundDownTree)) continue;


        /**
         *----s is the root of a segment tree that can be rounded down. Round down s.
         */
        s->cHint = s->cHintLow;


        /**
         *----Set the canRoundDownTree and canRoundDownSolo Booleans of s to false
         *----(i.e., s can only be rounded down once)
         */
        s->canRoundDownTree = 0;
        s->canRoundDownSolo = 0;


        /**
         *----Append the children of s to L and NULL-terminate L
         */
        for (i = 0; i < s->numChildren; ++i) {
            last->link = s->children[i];
            last = last->link;
        }
        last->link = 0;
    }
}

static ADF_Void MAZStrokeRoundUpTree (MAZStrokeSegment *s)
{
    /*----Initialize last to s. last represents a pointer to the last element in the
     *----linked list L (see below).
     */
    MAZStrokeSegment *last = s;


    /*----Initialize the link pointer of s to NULL. link is used below to implement a
     *----linked list of segments (denoted L below) for performing a breadth-first
     *----traversal of the segment tree whose root is s.
    */
    s->link = 0;


    /*----Process the segment tree using a breadth-first traversal of L*/
    for (; s; s = s->link) {


        /*----If s is the root of a segment tree that cannot be rounded down (i.e.,
         *----the segment tree has already been rounded down and therefore cannot be
         *----rounded down again), then proceed to the next segment
        */
        ADF_I32 i;
        if (!(s->canRoundDownTree)) continue;


        /*----s is the root of a segment tree that can be rounded down. Round down s.*/
        s->cHint = s->cHintLow;


        /*----Set the canRoundDownTree and canRoundDownSolo Booleans of s to false
         *----(i.e., s can only be rounded down once)
        */
        s->canRoundDownTree = 0;
        s->canRoundDownSolo = 0;


        /*----Append the children of s to L and NULL-terminate L*/
        for (i = 0; i < s->numChildren; ++i) {
            last->link = s->children[i];
            last = last->link;
        }
        last->link = 0;
    }
}

static ADF_Void MAZStrokesShiftDownHints (MAZStrokeSegment *sp)
{

    ADF_I32 j;

    if (!sp->numParents) {
        sp->cHint--;
        return;
    }

    sp->cHint--;
    for (j = 0; j < sp->numParents; ++j) {
        MAZStrokeSegment *parent = sp->parents[j];

        if (parent->cHint > sp->cHint)
            MAZStrokesShiftDownHints(sp->parents[j]);


        else if (sp->cHint - parent->cHint <= 1) 
            MAZStrokesShiftDownHints(sp->parents[j]);

    }


}

static ADF_Void MAZStrokesShiftUpHints (MAZStrokeSegment *sp)
{

    ADF_I32 j;

    if (sp->canRoundDownSolo) {
        sp->cHint++;
        sp->canRoundDownTree--;
        sp->canRoundDownSolo = 0;
        return;
    }
    if (sp->canRoundDownTree) {
        sp->cHint++;
        sp->canRoundDownTree--;
        for (j = 0; j < sp->numChildren; ++j) {
            MAZStrokesShiftUpHints(sp->children[j]);
        }
    }


}



/**
 *-----------------------------------------------------------------------------------
 *    Update the hinted coordinates of the segments in the specified segment array segs
 *    to resolve colliding segments (see Steps 5a through 5e in the Algorithm Overview
 *    section for details). Upon entry, segs is a contiguous array of numSegs segments
 *    in ascending order.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeResolveCollisionsBottomUp (MAZStrokeSegment *segs,
                                                    ADF_I32 numSegs)
{
    /**
     *----Set a pointer to the first segment of segs
     */
    MAZStrokeSegment *s = segs;


    /**
     *----Detect and resolve collisions using a greedy bottom-up traversal of the
     *----segments (see the Algorithm Overview section for details)
     */
    ADF_I32 i;
    for (i = 0; i < numSegs; ++i, ++s) {


        /**
         *----Uninitialized local variables
         */
        ADF_I32 j;
        MAZStrokeSegment *maxChild;


        /**
         *----Initialize the canRoundDownTree Boolean of s to true (i.e., assume
         *----initially that s is the root of a segment tree that can be rounded down
         *----without creating any collisions whatsoever in the segment tree)
         */
        s->canRoundDownTree = 1;


        /**
         *----Determine if s has no children
         */
        if (!(s->numChildren)) {


            /**
             *----s has no children. Set its canRoundDownSolo Boolean to true if and
             *----only if s is rounded up.
             */
            s->canRoundDownSolo = (s->cHint > s->cHintLow);


            /**
             *----Proceed to the next segment
             */
            continue;
        }


        /**
         *----s has at least one child. Process each child of s.
         */
        for (j = 0; j < s->numChildren; ++j) {


            /**
             *----Set a pointer to the current child of s
             */
            MAZStrokeSegment *child = s->children[j];


            /**
             *----Determine if the current child collides with s (i.e., the current
             *----child is exactly 1 pixel below s)
             */
            if (s->cHint - child->cHint == 1) {


                /**
                 *----The current child collides with s. Determine if the current
                 *----child can be rounded down.
                 */
                if (child->canRoundDownSolo) {


                    /**
                     *----The current child can be rounded down without colliding
                     *----with any of its children. Round down the current child,
                     *----thereby moving the current child 1 pixel farther away from
                     *----s. Set the current child's canRoundDownSolo and
                     *----canRoundDownTree Booleans to false (i.e., the current child
                     *----has just been rounded down and cannot be rounded down
                     *----again). Proceed to the next child.
                     */
                    child->cHint = child->cHintLow;
                    child->canRoundDownSolo = 0;
                    child->canRoundDownTree = 0;
                    continue;
                }


                /**
                 *----The current child cannot be rounded down without colliding with
                 *----at least one of its children. Determine if the current child is
                 *----the root of a segment tree that can be rounded down.
                 */
                if (child->canRoundDownTree) {


                    /**
                     *----The current child is the root of a segment tree that can be
                     *----rounded down. Round down the segment tree (i.e., the
                     *----current child and all of its children), thereby moving the
                     *----current child 1 pixel farther away from s. Proceed to the
                     *----next child.
                     */
                    MAZStrokeRoundDownTree(child);
                    continue;
                }


                /**
                 *----The current child is not the root of a segment tree that can be
                 *----rounded down. If s is rounded down, then round up s, thereby
                 *----moving s 1 pixel farther away from the current child. If s is
                 *----already rounded up, then this collision cannot be resolved by
                 *----rounding. This collision will be resolved by merging (see
                 *----below).
                 */
                if (s->cHint == s->cHintLow) ++(s->cHint);
            }
        }


        /**
         *----Initialize maxChild to the first child of s
         */
        maxChild = s->children[0];


        /**
         *----Process each child of s to determine maxChild, the child of s with the
         *----maximum hinted coordinate, and to determine the canRoundDownTree
         *----Boolean of s. maxChild will be used after the following loop terminates
         *----for subsequent processing (e.g., to determine if s and maxChild should
         *----be merged).
         */
        for (j = 0; j < s->numChildren; ++j) {


            /**
             *----Set a pointer to the current child of s
             */
            MAZStrokeSegment *child = s->children[j];


            /**
             *----Update maxChild if the current child has a greater hinted
             *----coordinate than maxChild
             */
            if (child->cHint > maxChild->cHint) maxChild = child;


            /**
             *----If the current child is the root of a segment tree that cannot be
             *----rounded down or rounding down both s and the current child would
             *----cause them to collide, then the segment tree whose root is s cannot
             *----be rounded down
             */
            if (!(child->canRoundDownTree) || (s->cHintLow - child->cHintLow <= 1))
                s->canRoundDownTree = 0;
        }


        /**
         *----Determine if s collides with maxChild
         */
        if ((s->cHint - maxChild->cHint) == 1) {


            /**
             *----s collides with maxChild. This collision cannot be resolved by
             *----rounding (see above). Resolve this collision by aligning s to
             *----maxChild and setting its isCollapsed Boolean to true.
             */
            s->cHint = maxChild->cHint;
            s->isCollapsed = 1;


            /**
             *----Set the canRoundDownSolo and canRoundDownTree Booleans of s to
             *----false to prevent s from being rounded (or adjusted in any way) in a
             *----subsequent iteration of this for-loop
             */
            s->canRoundDownSolo = 0;
            s->canRoundDownTree = 0;
            

        } else {


            /**
             *----s does not collide with maxChild. s can be rounded down if and only
             *----if (1) s is currently rounded up and (2) rounding down s does not
             *----cause s to collide with maxChild.
             */
            s->canRoundDownSolo = ((s->cHint > s->cHintLow) && (s->cHintLow -
            maxChild->cHint > 1));
        }
    }
}

static ADF_Void MAZStrokeResolveCollisionsTopDown (MAZStrokeSegment *segs,
                                                   ADF_I32 numSegs,
                                                   ADF_I32 *foundacollision)
{
    /*----Set a pointer to the first segment of segs*/
    MAZStrokeSegment *s = segs;


    /*----Detect and resolve collisions using a greedy bottom-up traversal of the
     *----segments (see the Algorithm Overview section for details)
    */
    ADF_I32 i;
    for (i = 0; i < numSegs; ++i, ++s) {


        /*----Uninitialized local variables*/
        ADF_I32 j;
        MAZStrokeSegment *maxChild;


        /*----Initialize the canRoundDownTree Boolean of s to true (i.e., assume
         *----initially that s is the root of a segment tree that can be rounded down
         *----without creating any collisions whatsoever in the segment tree)
         */
        s->canRoundDownTree = 1;


        /*----Determine if s has no children*/
        if (!(s->numChildren)) {


            /*----s has no children. Set its canRoundDownSolo Boolean to true if and
             *----only if s is rounded up.
             */
            s->canRoundDownSolo = (s->cHint < s->cHintLow);


            /*----Proceed to the next segment*/
            continue;
        }


        /*----s has at least one child. Process each child of s.*/
        for (j = 0; j < s->numChildren; ++j) {


            /*----Set a pointer to the current child of s*/
            MAZStrokeSegment *child = s->children[j];


            /*----Determine if the current child collides with s (i.e., the current
             *----child is exactly 1 pixel below s)
             */
            if (s->cHint - child->cHint == -1) {

                *foundacollision = 1;

                /*----The current child collides with s. Determine if the current
                 *----child can be rounded down.
                 */
                if (child->canRoundDownSolo) {


                    /*----The current child can be rounded down without colliding
                     *----with any of its children. Round down the current child,
                     *----thereby moving the current child 1 pixel farther away from
                     *----s. Set the current child's canRoundDownSolo and
                     *----canRoundDownTree Booleans to false (i.e., the current child
                     *----has just been rounded down and cannot be rounded down
                     *----again). Proceed to the next child.
                     */
                    child->cHint = child->cHintLow;
                    child->canRoundDownSolo = 0;
                    child->canRoundDownTree = 0;
                    continue;
                }


                /*----The current child cannot be rounded down without colliding with
                 *----at least one of its children. Determine if the current child is
                 *----the root of a segment tree that can be rounded down.
                 */
                if (child->canRoundDownTree) {


                    /*----The current child is the root of a segment tree that can be
                     *----rounded down. Round down the segment tree (i.e., the
                     *----current child and all of its children), thereby moving the
                     *----current child 1 pixel farther away from s. Proceed to the
                     *----next child.
                     */
                    MAZStrokeRoundUpTree(child);
                    continue;
                }


                /*----The current child is not the root of a segment tree that can be
                 *----rounded down. If s is rounded down, then round up s, thereby
                 *----moving s 1 pixel farther away from the current child. If s is
                 *----already rounded up, then this collision cannot be resolved by
                 *----rounding. This collision will be resolved by merging (see
                 *----below).
                 */
                if (s->cHint == s->cHintLow) --(s->cHint);
            }
        }


        /*----Initialize maxChild to the first child of s*/
        maxChild = s->children[0];


        /*----Process each child of s to determine maxChild, the child of s with the
         *----maximum hinted coordinate, and to determine the canRoundDownTree
         *----Boolean of s. maxChild will be used after the following loop terminates
         *----for subsequent processing (e.g., to determine if s and maxChild should
         *----be merged).
         */
        for (j = 0; j < s->numChildren; ++j) {


            /*----Set a pointer to the current child of s*/
            MAZStrokeSegment *child = s->children[j];


            /*----Update maxChild if the current child has a greater hinted
             *----coordinate than maxChild
             */
            if (child->cHint < maxChild->cHint) maxChild = child;


            /*----If the current child is the root of a segment tree that cannot be
             *----rounded down or rounding down both s and the current child would
             *----cause them to collide, then the segment tree whose root is s cannot
             *----be rounded down
             */
            if (!(child->canRoundDownTree) || (child->cHintLow - s->cHintLow <= 1))
                s->canRoundDownTree = 0;
        }


        /*----Determine if s collides with maxChild*/
        if ((s->cHint - maxChild->cHint) == -1) {


            /*----s collides with maxChild. This collision cannot be resolved by
             *----rounding (see above). Resolve this collision by aligning s to
             *----maxChild and setting its isCollapsed Boolean to true.
             */
            s->cHint = maxChild->cHint;
            s->isCollapsed = 1;

            *foundacollision = 1;


            /*----Set the canRoundDownSolo and canRoundDownTree Booleans of s to
             *----false to prevent s from being rounded (or adjusted in any way) in a
             *----subsequent iteration of this for-loop
             */
            s->canRoundDownSolo = 0;
            s->canRoundDownTree = 0;


        } else {


            /*----s does not collide with maxChild. s can be rounded down if and only
             *----if (1) s is currently rounded up and (2) rounding down s does not
             *----cause s to collide with maxChild.
             */
            s->canRoundDownSolo = ((s->cHint < s->cHintLow) &&
                (s->cHintLow - maxChild->cHint < -1));
        }
    }
}
static ADF_Void MAZStrokeResolveCollisionsWithHints(MAZStrokeSegment *segPairs,
                                                ADF_I32 numSegPairs,
                                                ADF_I32 *foundacollision)
{

    MAZStrokeSegment *sp = segPairs;
    ADF_I32 j;
    MAZStrokeSegment *maxChild;


    /**
     *----Detect and resolve collisions using a greedy bottom-up traversal of the
     *----segment pairs (see the Algorithm Overview section for details)
     */
    ADF_I32 i;

    *foundacollision = 1;

    for (i = 0; i < numSegPairs; ++i, ++sp) {

        if (!(sp->hintFlags&HF_HORVREDUCE)) 
            continue;



        /* found the next stroke to collapse */

        for (j = 0; j < sp->numParents; ++j) {


            /**
             *----Set a pointer to the current child of sp
             */
            MAZStrokeSegment *par = sp->parents[j];


            par->cHint = sp->cHint;


        }

    }

    sp = segPairs;

    for (i = 0; i < numSegPairs; ++i, ++sp) {

        /* found the next stroke to collapse */
        if (sp->numChildren == 0) {
            /* never move the top stroke */
            sp->canRoundDownSolo = 0;
            sp->canRoundDownTree = 0;
            continue;
        }


        /**
         *----Initialize maxChild to the first child of sp
         */
        maxChild = sp->children[0];
        sp->canRoundDownSolo = 0;
        sp->canRoundDownTree = 999;

        /**
         *----Process each child of sp to determine maxChild, the child of sp with
         *----the maximum hinted coordinate, and to determine the canRoundDownTree
         *----Boolean of sp. maxChild will be used after the following loop
         *----terminates for subsequent processing (e.g., to determine if sp and
         *----maxChild should be merged).
         */
        for (j = 0; j < sp->numChildren; ++j) {


            MAZStrokeSegment *child = sp->children[j];




            if (child->cHint < maxChild->cHint) maxChild = child;

            /* find minimum # of spaces in child trees */
            if (sp->canRoundDownTree > child->canRoundDownTree)
                sp->canRoundDownTree = child->canRoundDownTree;


        }

        if (maxChild->cHint - sp->cHint > 2) {
            sp->canRoundDownSolo = 1;
            sp->canRoundDownTree++;

        }

        if (maxChild->cHint - sp->cHint == 1) {
            /* have a collision */
            if (sp->canRoundDownTree) {
                MAZStrokesShiftUpHints(sp);
                sp->cHint--;  /* currently, rounddownhints moves sp down as well.*/
            }
            else
                MAZStrokesShiftDownHints(sp);


        }
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    Return true (i.e., a non-zero value) if the non-negative real-valued image
 *    coordinates a and b are within MAZ_STROKE_ONE_FIFTH units of each other. Return
 *    false otherwise.
 *-----------------------------------------------------------------------------------
 */
#define MAZ_STROKE_EQ(a,b) (((a) > (b)) ? ((a) - (b) < MAZ_STROKE_ONE_FIFTH) : \
((b) - (a) < MAZ_STROKE_ONE_FIFTH))




/**
 *-----------------------------------------------------------------------------------
 *    Upon entry, orig is a non-negative real-valued image coordinate, segs is a
 *    contiguous array of numSegs segments in ascending order, and snapMin is a
 *    non-negative real-valued image coordinate. MAZStrokeGetMinAnchor() attempts to
 *    find the minimum anchor segment of orig (i.e., the non-collapsed segment in the
 *    segs array with the maximum original coordinate that is less than or equal to
 *    orig). If the minimum anchor segment is found, the original and hinted
 *    coordinates of the minimum anchor segment are stored in outOrig and outSnap,
 *    respectively. If no minimum anchor segment is found, snapMin is stored in both
 *    outOrig and outSnap.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeGetMinAnchorTopDown (ADF_R32 orig, MAZStrokeSegment *segs,
                                              ADF_I32 numSegs, ADF_R32 snapMin,
                                              ADF_R32 *outOrig, ADF_R32 *outSnap)
{
    /**
     *----Set a pointer to the first segment in segs
     */
    MAZStrokeSegment *s = segs + (numSegs - 1);



    /**
     *----Initialize the minimum anchor segment to NULL
     */
    MAZStrokeSegment *anchor = 0;


    /**
     *----Determine the minimum anchor segment
     */
    ADF_I32 i;
    for (i = numSegs - 1; i >= 0; --i, --s) {

        if (orig < s->cOrig) break;
        if (s->isCollapsed) continue;
        anchor = s;
    }


    /**
     *----If the minimum anchor segment exists, set outOrig to the minimum anchor
     *----segment's original coordinate and set outSnap to the minimum anchor
     *----segment's hinted coordinate. If no minimum anchor segment exists, set both
     *----outOrig and outSnap to snapMin.
     */
    if (anchor) {
        *outOrig = anchor->cOrig;
        *outSnap = MAZ_STROKE_INT_TO_R32(anchor->cHint);
    } else *outOrig = *outSnap = snapMin;
}


/**
 *-----------------------------------------------------------------------------------
 *    Upon entry, orig is a non-negative real-valued image coordinate, segs is a
 *    contiguous array of numSegs segments in ascending order, and snapMax is a
 *    non-negative real-valued image coordinate. MAZStrokeGetMaxAnchor() attempts to
 *    find the maximum anchor segment of orig (i.e., the non-collapsed segment in the
 *    segs array with the minimum original coordinate that is greater than or equal to
 *    orig). If the maximum anchor segment is found, the original and hinted
 *    coordinates of the maximum anchor segment are stored in outOrig and outSnap,
 *    respectively. If no maximum anchor segment is found, snapMax is stored in both
 *    outOrig and outSnap.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeGetMaxAnchorTopDown (ADF_R32 orig, MAZStrokeSegment *segs,
                                              ADF_I32 numSegs, ADF_R32 snapMax,
                                              ADF_R32 *outOrig, ADF_R32 *outSnap)
{
    /**
     *----Set a pointer to the last segment in segs
     */
    MAZStrokeSegment *s = segs;


    /**
     *----Initialize the maximum anchor segment to NULL
     */
    MAZStrokeSegment *anchor = 0;


    /**
     *----Determine the maximum anchor segment
     */
    ADF_I32 i;
    for (i = 0; i < numSegs; ++i, ++s) {
        if (orig > s->cOrig) break;
        if (s->isCollapsed) continue;
        anchor = s;
    }


    /**
     *----If the maximum anchor segment exists, set outOrig to the maximum anchor
     *----segment's original coordinate and set outSnap to the maximum anchor
     *----segment's hinted coordinate. If no maximum anchor segment exists, set both
     *----outOrig and outSnap to snapMax.
     */
    if (anchor) {
        *outOrig = anchor->cOrig;
        *outSnap = MAZ_STROKE_INT_TO_R32(anchor->cHint);
    } else *outOrig = *outSnap = snapMax;
}

/**
 *-----------------------------------------------------------------------------------
 *    Upon entry, orig is a non-negative real-valued image coordinate, segs is a
 *    contiguous array of numSegs segments in ascending order, and snapMin is a
 *    non-negative real-valued image coordinate. MAZStrokeGetMinAnchor() attempts to
 *    find the minimum anchor segment of orig (i.e., the non-collapsed segment in the
 *    segs array with the maximum original coordinate that is less than or equal to
 *    orig). If the minimum anchor segment is found, the original and hinted
 *    coordinates of the minimum anchor segment are stored in outOrig and outSnap,
 *    respectively. If no minimum anchor segment is found, snapMin is stored in both
 *    outOrig and outSnap.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeGetMinAnchorBottomUp (ADF_R32 orig, MAZStrokeSegment *segs,
                                               ADF_I32 numSegs, ADF_R32 snapMin,
                                               ADF_R32 *outOrig, ADF_R32 *outSnap)
{
    /**
     *----Set a pointer to the first segment in segs   
     */
    MAZStrokeSegment *s = segs;


    /**
     *----Initialize the minimum anchor segment to NULL
     */
    MAZStrokeSegment *anchor = 0;


    /**
     *----Determine the minimum anchor segment
     */
    ADF_I32 i;
    for (i = 0; i < numSegs; ++i, ++s) {
        if (orig < s->cOrig) break;
        if (s->isCollapsed) continue;
        anchor = s;
    }


    /**
     *----If the minimum anchor segment exists, set outOrig to the minimum anchor
     *----segment's original coordinate and set outSnap to the minimum anchor
     *----segment's hinted coordinate. If no minimum anchor segment exists, set both
     *----outOrig and outSnap to snapMin.
     */
    if (anchor) {
        *outOrig = anchor->cOrig;
        *outSnap = MAZ_STROKE_INT_TO_R32(anchor->cHint);
    } else *outOrig = *outSnap = snapMin;
}


/**
 *-----------------------------------------------------------------------------------
 *    Upon entry, orig is a non-negative real-valued image coordinate, segs is a
 *    contiguous array of numSegs segments in ascending order, and snapMax is a
 *    non-negative real-valued image coordinate. MAZStrokeGetMaxAnchor() attempts to
 *    find the maximum anchor segment of orig (i.e., the non-collapsed segment in the
 *    segs array with the minimum original coordinate that is greater than or equal to
 *    orig). If the maximum anchor segment is found, the original and hinted
 *    coordinates of the maximum anchor segment are stored in outOrig and outSnap,
 *    respectively. If no maximum anchor segment is found, snapMax is stored in both
 *    outOrig and outSnap.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeGetMaxAnchorBottomUp (ADF_R32 orig, MAZStrokeSegment *segs,
                                               ADF_I32 numSegs, ADF_R32 snapMax,
                                               ADF_R32 *outOrig, ADF_R32 *outSnap)
{
    /**
     *----Set a pointer to the last segment in segs
     */
    MAZStrokeSegment *s = segs + (numSegs - 1);


    /**
     *----Initialize the maximum anchor segment to NULL
     */
    MAZStrokeSegment *anchor = 0;


    /**
     *----Determine the maximum anchor segment
     */
    ADF_I32 i;
    for (i = numSegs - 1; i >= 0; --i, --s) {
        if (orig > s->cOrig) break;
        if (s->isCollapsed) continue;
        anchor = s;
    }


    /**
     *----If the maximum anchor segment exists, set outOrig to the maximum anchor
     *----segment's original coordinate and set outSnap to the maximum anchor
     *----segment's hinted coordinate. If no maximum anchor segment exists, set both
     *----outOrig and outSnap to snapMax.
     */
    if (anchor) {
        *outOrig = anchor->cOrig;
        *outSnap = MAZ_STROKE_INT_TO_R32(anchor->cHint);
    } else *outOrig = *outSnap = snapMax;
}




/**
 *-----------------------------------------------------------------------------------
 *    Perform grid fitting on the near-segment coordinates of the pen commands in the
 *    specified pen command array penCmds. Upon entry, hSegs and vSegs are contiguous
 *    arrays of numHSegs and numVSegs horizontal and vertical segments, respectively,
 *    and penCmds is the contiguous array of numPenCmds pen commands to be grid fit. If
 *    the x coordinate of a pen command P is modified, the MAZ_STROKE_X_ALIGNED bit of
 *    P's opCode element is set. Similarly, if the y coordinate of a pen command P is
 *    modified, the MAZ_STROKE_Y_ALIGNED bit of P's opCode element is set. Coordinate
 *    updates are performed in place (i.e., the original near-segment coordinates in
 *    penCmds are overwritten).
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeAutohintNearSegmentCoordinates (fnt_LocalGraphicStateType *gs,
                                                         MAZStrokeSegment *hSegs,
                                                         MAZStrokeSegment *vSegs,
                                                         ADF_I32 numHSegs,
                                                         ADF_I32 numVSegs)
{
    /**
     *----Process each pen command
     */
    ADF_I32 i;


    for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {



        /**
         *----Set a pointer to the first horizontal segment in hSegs and a pointer to
         *----the first vertical segment in vSegs
         */
        MAZStrokeSegment *hSeg = hSegs;
        MAZStrokeSegment *vSeg = vSegs;


        /**
         *----Initialize xAlign and yAlign to -1, indicating that the x and y
         *----coordinates of the current pen command p are not near-segment
         *----coordinates and therefore do not require modification by this function
         */
        ADF_I32 xAlign = -999;
        ADF_I32 yAlign = -999;


        /**
         *----Determine if p's y coordinate lies on or near a horizontal segment
         */
        ADF_I32 j;
        gs->CE0->f[i] = 0;
        for (j = 0; j < numHSegs; ++j, ++hSeg) {


            /**
             *----If p's y coordinate is sufficiently close to the original y
             *----coordinate of hSeg and p's x coordinate overlaps hSeg in the x
             *----direction, then set the MAZ_STROKE_Y_ALIGNED bit of p's opCode
             *----element, set yAlign to hSeg's hinted y coordinate, and exit the
             *----loop
             */
            if ((hSeg->cMin <= gs->CE0->x[i]) && (gs->CE0->x[i] <= hSeg->cMax) &&
            MAZ_STROKE_EQ(gs->CE0->y[i], hSeg->cOrig)) {
                gs->CE0->f[i] |= (MAZ_STROKE_Y_ALIGNED>>24);
                yAlign = hSeg->cHint;
                break;
            }
        }


        /**
         *----Determine if p's x coordinate lies on or near a vertical segment
         */
        for (j = 0; j < numVSegs; ++j, ++vSeg) {


            /**
             *----If p's x coordinate is sufficiently close to the original x
             *----coordinate of vSeg and p's y coordinate overlaps vSeg in the y
             *----direction, then set the MAZ_STROKE_X_ALIGNED bit of p's opCode
             *----element, set xAlign to vSeg's hinted x coordinate, and exit the
             *----loop
             */
            if ((vSeg->cMin <= gs->CE0->y[i]) && (gs->CE0->y[i] <= vSeg->cMax) &&
            MAZ_STROKE_EQ(gs->CE0->x[i], vSeg->cOrig)) {
                gs->CE0->f[i] |= (MAZ_STROKE_X_ALIGNED>>24);
                xAlign = vSeg->cHint;
                break;
            }
        }


        /**
         *----If yAlign is valid, set p's hinted y coordinate to yAlign
         */
        if (yAlign >= -998) gs->CE0->y[i] = MAZ_STROKE_INT_TO_R32(yAlign);


        /**
         *----If xAlign is valid, set p's hinted x coordinate to xAlign
         */
        if (xAlign >= -998) gs->CE0->x[i] = MAZ_STROKE_INT_TO_R32(xAlign);
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    Perform grid fitting on the non-segment coordinates of the pen commands in the
 *    specified pen command array penCmds. Upon entry, hSegs and vSegs are contiguous
 *    arrays of numHSegs and numVSegs horizontal and vertical segments in ascending
 *    order, respectively. penCmds is the contiguous array of numPenCmds pen commands
 *    to be grid fit. xMin, yMin, xMax, and yMax are the glyph's original bounding box
 *    coordinates (in real-valued image coordinates). Each non-segment coordinate of
 *    each pen command in penCmds is grid fit in-place (i.e., the original non-segment
 *    coordinates in penCmds are overwritten).
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeAutohintNonSegmentCoordinates (fnt_LocalGraphicStateType *gs,
                                  MAZStrokeSegment *hSegs,MAZStrokeSegment *vSegs,
                                  ADF_I32 numHSegs, ADF_I32 numVSegs,
                                  ADF_R32 xMin, ADF_R32 yMin, ADF_R32 xMax, ADF_R32 yMax)
{
    /**
     *----Process each pen command
     */
    ADF_I32 i;

    for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {



        /**
         *----If p->y has not already been grid fit in the y direction, grid fit p->y
         *----in the y direction
         */
        if (!(gs->CE0->f[i] & (MAZ_STROKE_Y_ALIGNED>>24))) {


            /**
             *----Obtain the minimum and maximum y anchor coordinates (i.e., the
             *----original and hinted y coordinates of the minimum and maximum anchor
             *----segments of p->y, respectively). If the minimum anchor segment is
             *----missing, the bottom edge of the glyph's bounding box (i.e., yMin)
             *----is used as a "virtual" minimum anchor segment. If the maximum
             *----anchor segment is missing, the top edge of the glyph's bounding box
             *----(i.e., yMax) is used as a "virtual" maximum anchor segment.
             */
            ADF_R32 loOrig, loSnap;
            ADF_R32 hiOrig, hiSnap;

            if (gs->globalGS->maz_data.MAZmethodHorz == MAZBOTTOMUP) {
                MAZStrokeGetMinAnchorBottomUp(gs->CE0->y[i], hSegs,
                numHSegs, yMin, &loOrig, &loSnap);
                MAZStrokeGetMaxAnchorBottomUp(gs->CE0->y[i], hSegs,
                numHSegs, yMax, &hiOrig, &hiSnap);
            }
            else {
                MAZStrokeGetMinAnchorTopDown(gs->CE0->y[i], hSegs,
                numHSegs, yMin, &loOrig, &loSnap);
                MAZStrokeGetMaxAnchorTopDown(gs->CE0->y[i], hSegs,
                numHSegs, yMax, &hiOrig, &hiSnap);
            }


            /**
             *----If the original coordinates of the minimum and maximum anchor
             *----segments are different, then perform linear interpolation to
             *----determine p's hinted y coordinate. Otherwise, set p's hinted y
             *----coordinate to the hinted coordinate of the minimum anchor segment.
             */
            if (hiOrig != loOrig) gs->CE0->y[i] = MAZStrokeLerp(gs->CE0->y[i],
            loOrig, hiOrig, loSnap, hiSnap);
            else gs->CE0->y[i] = loSnap;
        }


        /**
         *----If p->x has not already been grid fit in the x direction, grid fit p->x
         *----in the x direction
         */
        if (!(gs->CE0->f[i] & (MAZ_STROKE_X_ALIGNED>>24))) {


            /**
             *----Obtain the minimum and maximum x anchor coordinates (i.e., the
             *----original and hinted x coordinates of the minimum and maximum anchor
             *----segments of p->x, respectively). If the minimum anchor segment is
             *----missing, the left edge of the glyph's bounding box (i.e., xMin) is
             *----used as a "virtual" minimum anchor segment. If the maximum anchor
             *----segment is missing, the right edge of the glyph's bounding box
             *----(i.e., xMax) is used as a "virtual" maximum anchor segment.
             */
            ADF_R32 loOrig, loSnap;
            ADF_R32 hiOrig, hiSnap;
            if (gs->globalGS->maz_data.MAZmethodVert == MAZBOTTOMUP) {
                MAZStrokeGetMinAnchorBottomUp(gs->CE0->x[i], vSegs,
                numVSegs, xMin, &loOrig, &loSnap);
                MAZStrokeGetMaxAnchorBottomUp(gs->CE0->x[i], vSegs,
                numVSegs, xMax, &hiOrig, &hiSnap);
            }
            else {
                MAZStrokeGetMinAnchorTopDown(gs->CE0->x[i], vSegs,
                numVSegs, xMin, &loOrig, &loSnap);
                MAZStrokeGetMaxAnchorTopDown(gs->CE0->x[i], vSegs,
                numVSegs, xMax, &hiOrig, &hiSnap);
            }


            /**
             *----If the original coordinates of the minimum and maximum anchor
             *----segments are different, then perform linear interpolation to
             *----determine p's hinted x coordinate. Otherwise, set p's hinted x
             *----coordinate to the hinted coordinate of the minimum anchor segment.
             */
            if (hiOrig != loOrig) gs->CE0->x[i] = MAZStrokeLerp(gs->CE0->x[i],
            loOrig, hiOrig, loSnap, hiSnap);
            else gs->CE0->x[i] = loSnap;
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    Determine the hinted path width by rounding the specified original path width
 *    pathWidth to the nearest integer (see details below) and ensuring that the hinted
 *    result is at least 1 pixel. If the hinted path width is an even integer, stroke
 *    skeletons in penCmds will be aligned to the half-integer pixel grid (the stroke
 *    skeletons specified in penCmds are aligned to the integer pixel grid). Upon
 *    entry, penCmds is the contiguous array of numPenCmds pen commands and pathWidth
 *    contains the original path width (specified in real-valued image coordinates) to
 *    be grid fit. Upon exit, pathWidth contains the hinted path width (i.e., the
 *    original path width is overwritten). Also note that upon exit, if the hinted path
 *    width is an even integer, the coordinates of penCmds will have changed to
 *    accommodate the hinted path width.
 *
 *    Consider the scenario where multiple glyphs, all specified with an identical path
 *    width in floating point font units, are rendered to a single page of text
 *    displayed on a screen. Because each glyph maintains its own transformation for
 *    transforming coordinates and units from floating point font units to real-valued
 *    image coordinates, slight differences in the transformed real-valued path widths
 *    arise because of numerical round-off errors that occur during the sequence of
 *    steps in which the path widths are transformed from floating point font units to
 *    floating point ADF coordinates and then to real-valued image coordinates. If the
 *    transformed path widths are simply rounded to the nearest integer, then some
 *    glyphs will have their hinted path widths set to an integer N whereas other
 *    glyphs will have their hinted path widths set to N+1. In a single page of text
 *    which displays both sets of glyphs, these inconsistencies in the rendered stroke
 *    weights are obvious and unacceptable. To avoid this problem, this implementation
 *    rounds down the transformed path width when its fractional portion is less than
 *    MAZ_MAX_SAFETY_THRESHOLD, a constant (chosen empirically) whose mathematical
 *    value is slightly higher than 0.5. The transformed path width is rounded up when
 *    its fractional portion is greater than or equal to MAZ_MAX_SAFETY_THRESHOLD.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MAZStrokeAutohintPathWidth (fnt_LocalGraphicStateType *gs,
                                            ADF_R32 *pathWidth)
{
    ADF_I32 i;


    /**
     *----Round the original path width to the nearest integer not greater than the
     *----original path width (e.g., if pathWidth has the mathematical value 1.3,
     *----then floorPathWidth will be set to the mathematical value of 1.0)
     */
    ADF_R32 floorPathWidth = MAZ_STROKE_FLOOR(*pathWidth);


    /**
     *----Determine the fractional portion of the original path width (e.g., if
     *----pathWidth has the mathematical value 1.3, then fracWidth has the
     *----mathematical value 1.3 - 1.0 = 0.3)
     */
    ADF_R32 fracWidth = *pathWidth - floorPathWidth;


    /**
     *----If the fractional portion of the original path width is less than
     *----MAZ_MAX_SAFETY_THRESHOLD, round down the original path width. Otherwise,
     *----round up the original path width.
     */
    if (fracWidth < MAZ_MAX_SAFETY_THRESHOLD) *pathWidth = floorPathWidth;
    else *pathWidth = floorPathWidth + MAZ_STROKE_ONE;


    /**
     *----Ensure that the hinted path width is at least 1 pixel
     */
    if (*pathWidth < MAZ_STROKE_ONE) *pathWidth = MAZ_STROKE_ONE;


    /**
     *----If the hinted path width is an even integer, add the mathematical value 0.5
     *----to each coordinate of each pen command. This will cause horizontal and
     *----vertical stroke skeletons to be aligned to the half-integer pixel grid.
     */
    if (!MAZ_STROKE_IS_EVEN(*pathWidth)) {
        
        for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {
            gs->CE0->x[i] += MAZ_STROKE_ONE_HALF;
            gs->CE0->y[i] += MAZ_STROKE_ONE_HALF;

        }
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    ADFAutohintMAZStrokeInternal() performs MAZ alignment zone detection and grid
 *    fitting on the specified sequence of pen commands penCmds representing a
 *    uniform-width stroke-based glyph. Although ADFAutohintMAZStrokeInternal() can be
 *    invoked on any uniform-width stroke-based glyph, its algorithms are designed and
 *    optimized for CJK glyphs. Upon entry, libInst is a valid Saffron library
 *    instance, penCmds is the contiguous array of numPenCmds pen commands to be grid
 *    fit, and pathWidth is a pointer to the path width (i.e., stroke width) of the
 *    uniform-width stroke-based glyph. xMin, yMin, xMax, and yMax are the glyph's
 *    bounding box coordinates (which account for the path width). The coordinates of
 *    the pen commands, the path width, and the bounding box must be specified in
 *    non-negative real-valued image coordinates. Grid fitting is performed in-place
 *    (i.e., the computed results are written to the x, y, cx, and cy coordinates of
 *    each pen command in the penCmds array, thereby overwriting their original
 *    values).
 *
 *    Upon exit, pathWidth contains the hinted path width determined during grid
 *    fitting, outNumHSegs contains the number of horizontal segments determined during
 *    grid fitting, outNumVSegs contains the number of vertical segments determined
 *    during grid fitting, outHSegs points to a contiguous array of outNumHSegs
 *    horizontal segments determined during grid fitting, outVSegs points to a
 *    contiguous array of outNumVSegs vertical segments determined during grid fitting,
 *    and outMem points to a contiguous block of memory that must be freed by the
 *    caller. Freeing outMem also frees outHSegs and outVSegs, thereby invalidating all
 *    three pointers. The caller should not free outHSegs or outVSegs directly.
 *
 *    ADFAutohintMAZStrokeInternal() returns true (i.e., a non-zero value) on success,
 *    false (i.e., zero) on failure. Upon failure, no changes are made to pathWidth or
 *    to the coordinates in the penCmds array, and no data is written to outNumHSegs,
 *    outNumVSegs, outHSegs, outVSegs, or outMem.
 *-----------------------------------------------------------------------------------
 */
ADF_I32 ADFAutohintMAZStrokeInternal (fnt_LocalGraphicStateType *gs, ADF_R32 strokeWidth,
                             ADF_R32 xMin, ADF_R32 yMin, ADF_R32 xMax, ADF_R32 yMax,
                             ADF_I32 *outNumHSegs, ADF_I32 *outNumVSegs,
                             MAZStrokeSegment **outHSegs,MAZStrokeSegment **outVSegs)
{
    /**
     *----Uninitialized local variables
     */

    ADF_I32 numHSegs;
    ADF_I32 numVSegs;
    MAZStrokeSegment *hSegs;
    MAZStrokeSegment *vSegs;



    /**
     *----Partition the single block of memory evenly into two arrays, one to store
     *----horizontal segments and one to store vertical segments
     */
    hSegs = (MAZStrokeSegment *) gs->globalGS->maz_data.hSegs;
    vSegs = (MAZStrokeSegment *) gs->globalGS->maz_data.vSegs;




    {

        /*----Determine the horizontal and vertical segments*/
        if (gs->globalGS->maz_data.MAZmethodVert == MAZTOPDOWN ||
            gs->globalGS->maz_data.MAZmethodVert == MAZUSEHINTS)
            MAZStrokeFindSegmentsTopDown(gs, hSegs, vSegs, &numHSegs, &numVSegs);
        else
            MAZStrokeFindSegmentsBottomUp(gs, hSegs, vSegs, &numHSegs, &numVSegs);


        /*----Sort the horizontal segments into ascending order. Similarly, sort the
         *----vertical segments into ascending order.
         */
        if (gs->globalGS->maz_data.MAZmethodHorz == MAZTOPDOWN ||
            gs->globalGS->maz_data.MAZmethodHorz == MAZUSEHINTS)
            MAZStrokeSortSegmentsTopDown(hSegs, numHSegs);
        else
            MAZStrokeSortSegmentsBottomUp(hSegs, numHSegs);

        if (gs->globalGS->maz_data.MAZmethodVert == MAZTOPDOWN ||
            gs->globalGS->maz_data.MAZmethodVert == MAZUSEHINTS)
            MAZStrokeSortSegmentsTopDown(vSegs, numVSegs);
        else
            MAZStrokeSortSegmentsBottomUp(vSegs, numVSegs);


        /*----Build a directed acyclic graph (DAG) for horizontal segments in which each
         *----horizontal segment is linked to its horizontal child segments. Similarly,
         *----build a DAG for vertical segments in which each vertical segment is linked
         *----to its vertical child segments.
         */
        if (gs->globalGS->maz_data.MAZmethodHorz == MAZTOPDOWN ||
            gs->globalGS->maz_data.MAZmethodHorz == MAZUSEHINTS)
            MAZStrokeBuildDagTopDown(hSegs, numHSegs);
        else
            MAZStrokeBuildDagBottomUp(hSegs, numHSegs);

        if (gs->globalGS->maz_data.MAZmethodVert == MAZTOPDOWN ||
            gs->globalGS->maz_data.MAZmethodVert == MAZUSEHINTS)
            MAZStrokeBuildDagTopDown(vSegs, numVSegs);
        else
            MAZStrokeBuildDagBottomUp(vSegs, numVSegs);




        /*----Detect and resolve collisions among horizontal segments. Similarly, detect
         *----and resolve collisions among vertical segments.
         */

        if (gs->globalGS->maz_data.MAZmethodHorz == MAZTOPDOWN)
            MAZStrokeResolveCollisionsTopDown(hSegs, numHSegs,
                    &gs->globalGS->maz_data.foundacollisionhorz);

        else if (gs->globalGS->maz_data.MAZmethodHorz == MAZUSEHINTS)
                MAZStrokeResolveCollisionsWithHints(hSegs, numHSegs,
                    &gs->globalGS->maz_data.foundacollisionhorz);
        else
            MAZStrokeResolveCollisionsBottomUp(hSegs, numHSegs);


        if (gs->globalGS->maz_data.MAZmethodVert == MAZTOPDOWN)
            MAZStrokeResolveCollisionsTopDown(vSegs, numVSegs,
                &gs->globalGS->maz_data.foundacollisionvert);
        else if (gs->globalGS->maz_data.MAZmethodVert == MAZUSEHINTS)
           MAZStrokeResolveCollisionsWithHints(vSegs, numVSegs,
                &gs->globalGS->maz_data.foundacollisionvert);
        else
            MAZStrokeResolveCollisionsBottomUp(vSegs, numVSegs);


    }


    /**
     *----Perform grid fitting on near-segment coordinates
     */
    MAZStrokeAutohintNearSegmentCoordinates(gs, hSegs, vSegs,
    numHSegs, numVSegs);


    /**
     *----Perform grid fitting on non-segment coordinates
     */
    MAZStrokeAutohintNonSegmentCoordinates(gs, hSegs, vSegs, numHSegs,
    numVSegs, xMin, yMin, xMax, yMax);


    /**
     *----Determine the hinted path width and adjust each pen command coordinate to
     *----accommodate the hinted path width if appropriate
     */
    MAZStrokeAutohintPathWidth(gs, &strokeWidth);


    /**
     *----Store the number of horizontal segments and the number of vertical segments
     */
    *outNumHSegs = numHSegs;
    *outNumVSegs = numVSegs;


    /**
     *----Store the pointer to the first horizontal segment and the pointer to the
     *----first vertical segment
     */
    *outHSegs = hSegs;
    *outVSegs = vSegs;





    /**
     *----Return success
     */
    return(1);
}

static ADF_Void SimplifyStrokesForBitmaps (fnt_LocalGraphicStateType *gs)
{
    ADF_I32 i;
    ADF_I32 j;

    ADF_R32 tol = 64;


    F26DOT6 *x = gs->CE0->x;
    F26DOT6 *y = gs->CE0->y;


    /*----Process each pen command */

    for (i = 0; i < gs->CE0->nc; ++i) {

        ADF_R32 xmin = x[gs->CE0->sp[i]];
        ADF_R32 xmax = x[gs->CE0->sp[i]];
        ADF_R32 ymin = y[gs->CE0->sp[i]];
        ADF_R32 ymax = y[gs->CE0->sp[i]];

        for (j=gs->CE0->sp[i]+1; j<=gs->CE0->ep[i]; j++) {
            if (x[j] < xmin)
                xmin = x[j];
            if (x[j] > xmax)
                xmax = x[j];
            if (y[j] < ymin)
                ymin = y[j];
            if (y[j] > ymax)
                ymax = y[j];

        }

        /* if entire contour is flat enough - completely flatten it */
        if (xmin != xmax && xmax-xmin < tol &&  ymax-ymin > 2*tol) {
            ADF_R32 average = (xmax+xmin)>>1;
            for (j=gs->CE0->sp[i]; j<=gs->CE0->ep[i]; j++) {
                x[j] = average;
            }
        }
        if (ymin != ymax && ymax-ymin < tol && xmax - xmin > 2*tol) {
            ADF_R32 average = (ymax+ymin)>>1;
            for (j=gs->CE0->sp[i]; j<=gs->CE0->ep[i]; j++) {
                y[j] = average;
            }
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    ADFAutohintMAZStroke() performs MAZ alignment zone detection and grid fitting on
 *    the specified sequence of pen commands penCmds representing a uniform-width
 *    stroke-based glyph. Although ADFAutohintMAZStroke() can be invoked on any
 *    uniform-width stroke-based glyph, its algorithms are designed and optimized for
 *    CJK glyphs. Upon entry, libInst is a valid Saffron library instance, penCmds is
 *    the contiguous array of numPenCmds pen commands to be grid fit, and pathWidth is
 *    a pointer to the path width (i.e., stroke width) of the uniform-width
 *    stroke-based glyph. xMin, yMin, xMax, and yMax are the glyph's bounding box
 *    coordinates (which account for the path width). The coordinates of the pen
 *    commands, the path width, and the bounding box must be specified in non-negative
 *    real-valued image coordinates. Grid fitting is performed in-place (i.e., the
 *    computed results are written to the x, y, cx, and cy coordinates of each pen
 *    command in the penCmds array, thereby overwriting their original values). Upon
 *    exit, pathWidth contains the hinted path width determined during grid fitting.
 *    ADFAutohintMAZStroke() returns true (i.e., a non-zero value) on success, false
 *    (i.e., zero) on failure. Upon failure, no changes are made to pathWidth or to the
 *    coordinates in the penCmds array.
 *-----------------------------------------------------------------------------------
 */
ADF_I32 ADFAutohintMAZStroke (fnt_LocalGraphicStateType *gs)
{
    /**
     *----Local variables
     */
    ADF_I32 numHSegs;
    ADF_I32 numVSegs;
    MAZStrokeSegment *hSegs;
    MAZStrokeSegment *vSegs;
    ADF_R32 xMin;
    ADF_R32 xMax;
    ADF_R32 yMin;
    ADF_R32 yMax;
    ADF_R32 strokeWidth = gs->globalGS->store[0] << 10;
    ADF_I32 i;
    ADF_I32 result;

    int adjustmentx;
    int adjustmenty;

    int oldadjustx;
    int oldadjusty;

    ADF_R32 ppemy;
    int above;
    int below;
    int left;
    int right;

    int pixelsonleft;
    int pixelsonright;
    int pixelsabove;
    int pixelsbelow;
    int newpixelsbelow;
    int newpixelsonleft;

    if (gs->CE0->nc == 0)
        return 0;

    SimplifyStrokesForBitmaps(gs);

    ppemy = gs->globalGS->pixelsPerEm;
    if ( !gs->globalGS->identityTransformation ) {

        ppemy = (ppemy * gs->globalGS->cvtStretchY + 32768)>>16;
    }

    above = (ppemy * 220 + 128)>>8;

    below = above - ppemy;

    gs->CE0->x[0] <<= 10;
    gs->CE0->y[0] <<= 10;

    xMin = gs->CE0->x[0];
    xMax = gs->CE0->x[0];
    yMin = gs->CE0->y[0];
    yMax = gs->CE0->y[0];

    for (i = 1; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {
        gs->CE0->x[i] <<= 10;
        gs->CE0->y[i] <<= 10;

        if (xMin > gs->CE0->x[i])
            xMin = gs->CE0->x[i];
        if (xMax < gs->CE0->x[i])
            xMax = gs->CE0->x[i];
        if (yMin > gs->CE0->y[i])
            yMin = gs->CE0->y[i];
        if (yMax < gs->CE0->y[i])
            yMax = gs->CE0->y[i];
    }

    if ( (above<<16) > yMax)
        yMax = above<<16;

    if ( (below<<16) < yMin)
        yMin = below<<16;


    for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {
        gs->CE0->x[i] -= xMin;
        gs->CE0->y[i] -= yMin;
    }

    oldadjusty = yMin>>10;
    oldadjustx = xMin>>10;
    xMax -= xMin;
    yMax -= yMin;
    xMin = 0;
    yMin = 0;



    for (i=0; i<gs->globalGS->maz_data.numHhints; i++) {
        gs->CE0->f[gs->globalGS->maz_data.Hcollapse[i]] |= 1;
        gs->globalGS->maz_data.MAZmethodHorz = MAZUSEHINTS;
    }

    for (i=0; i<gs->globalGS->maz_data.numVhints; i++) {
        gs->CE0->f[gs->globalGS->maz_data.Vcollapse[i]] |= 2;
        gs->globalGS->maz_data.MAZmethodVert = MAZUSEHINTS;
    }


    /**
     *----Perform MAZ alignment zone detection and grid fitting on the specified
     *----glyph
     */
    result = ADFAutohintMAZStrokeInternal(gs, strokeWidth, xMin, yMin, xMax, yMax,
                                          &numHSegs, &numVSegs, &hSegs, &vSegs);


    gs->CE0->x[0] >>= 10;
    gs->CE0->y[0] >>= 10;


    xMin = gs->CE0->x[0];
    xMax = gs->CE0->x[0];
    yMin = gs->CE0->y[0];
    yMax = gs->CE0->y[0];

    for (i = 1; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {


        gs->CE0->x[i] >>= 10;
        gs->CE0->y[i] >>= 10;

        if (xMin > gs->CE0->x[i])
            xMin = gs->CE0->x[i];
        if (xMax < gs->CE0->x[i])
            xMax = gs->CE0->x[i];
        if (yMin > gs->CE0->y[i])
            yMin = gs->CE0->y[i];
        if (yMax < gs->CE0->y[i])
            yMax = gs->CE0->y[i];
    }



    left = gs->CE0->x[gs->CE0->ep[gs->CE0->nc-1]+1];
    right = gs->CE0->x[gs->CE0->ep[gs->CE0->nc-1]+2];

    pixelsabove = (above) - ((yMax+31)>>6);
    pixelsbelow = ((yMin+31)>>6) - below;

    newpixelsbelow = (pixelsabove + pixelsbelow)>>1;

    adjustmenty = (newpixelsbelow - pixelsbelow) <<6;

    if (gs->globalGS->maz_data.noCenter )
        adjustmenty = (oldadjusty+32)&0xffffffc0;



    pixelsonleft = ((xMin+31)>>6) - ((left+31)>>6);
    pixelsonright = ((right+31)>>6) - ((xMax+31)>>6);

    newpixelsonleft = (pixelsonleft + pixelsonright)>>1;

    adjustmentx = (newpixelsonleft - pixelsonleft)<<6;

    if (gs->globalGS->maz_data.noCenter )
        adjustmentx = (oldadjustx+32)&0xffffffc0;



    for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) {
        gs->CE0->x[i] += adjustmentx;
        gs->CE0->y[i] += adjustmenty;
        gs->CE0->ox[i] = gs->CE0->x[i];
        gs->CE0->oy[i] = gs->CE0->y[i];
        gs->CE0->f[i] = 0;
    }

    /**
     *----Return the Boolean indicating success or failure
     */
    return(result);
}


/**
 *-----------------------------------------------------------------------------------
 *    END: MAZ DETECTION AND GRID FITTING FOR UNIFORM-WIDTH STROKE-BASED GLYPHS ONLY
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    END: IMPLICIT ADFS ONLY
 *-----------------------------------------------------------------------------------
 */
#endif

/**
 *-----------------------------------------------------------------------------------
 *    END: iType Edge Hinting
 *-----------------------------------------------------------------------------------
 */
#endif
